extensions [ array gis ]

breed [newts newt]

newts-own
[
  trigger           ; Determines the actions of a newt in the current month. Can take values "migrate-up" "migrate-pond" "breed" or "upland."
  home-pond-x       ; Newts remember their home pond location
  home-pond-y
  distance-to-pond  ; Keeps track of distance to home pond
  life-stage        ; Newts can be either efts or adults. Only adults can breed. Efts surviving for 6 months in upland metamorphose into adults. (this model disregards paedomorphs)
  rain-sens         ; The minimum rainfall in the current month required to cause a newt to return to pond. (distribution is user defined)
  breeding-counter  ; Counts the 6 months it takes a newt to reproduce in the pond. If pond depth drops to zero during this period, the newt fails to breed and returns to upland.
  total-breed-depth ; Keeps track of the depth of pond while newt is breeding. Reproduction is a function of average depth during breeding.
  eft-counter       ; Counts the six months it takes efts to mature into adults (in upland).
]

patches-own
[
  cover             ; percentage forest cover of upland patches
  depth             ; water depth in ponds
  cat-area-random         ; amount of area draining into each pond
  is-depression     ; whether patch is a depression
  pond-area         ; the area of ponds (in m2) under the real landscape scenario.
  cat-area          ; area of pond catchments (in m2) under the real landscape scenario.
  temp-cover        ; stores a randomly generated value with a maximum of the user-inputted forest-cover-multiplier modifier
  pond-area-random  ; The area (in patches) of ponds under the random landscape scenario. It is currently set to 1 patch by default.
]

globals
[
   rainfall          ; array that stores the rainfall data for the entire period
   ppt               ; variable for the current month's rainfall
   rain-data-points     ; number of values in each rainfall array
   pond-depths          ; this list stores depths of all ponds
   land-cover-dataset  
   ponds-dataset 
   catchment-dataset
   eft-proportion       ; The proportion of newts that are efts at model start
   precip-av            ; These variables track the six month average rainfalls for the purposes of calculating the precipitation-based mortality multiplier (see below)
   past-precip-1 
   past-precip-2
   past-precip-3
   past-precip-4
   past-precip-5
   mort-multiplier      ; Tracks the mortality adjustment resulting from two factors: 1) overall mortality adjustment (upland-mortality), 
                        ; and 2) the quotient of 6 month average rainfall and user-defined base rainfall amount.
]
   
to SETUP
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  setup-patches
  choose-rainfall
  setup-newts
  setup-newts-upland
  setup-efts
end

to setup-patches          ; User selects whether to use real landscape data or randomly generated landscape.
  ifelse Real_Landscape?
     [ setup-patches-from-gis ]
     [ setup-patches-random ]
end

to setup-patches-from-gis
  gis:load-coordinate-system ("input/elev.ft.prj")
  set land-cover-dataset gis:load-dataset "input/canopy_100_cubic.asc"
  set ponds-dataset      gis:load-dataset"input/ponds_100_3.asc"
  set catchment-dataset  gis:load-dataset"input/catchment_100_3.asc"
  gis:set-world-envelope (gis:envelope-union-of (gis:envelope-of land-cover-dataset)
                                                (gis:envelope-of ponds-dataset)
                                                (gis:envelope-of catchment-dataset))
  gis:apply-raster land-cover-dataset cover
  gis:apply-raster ponds-dataset pond-area
  gis:apply-raster catchment-dataset cat-area
  
  ask patches with [ cover >= 0 ]
    [ set pcolor (59 - cover / 13)]
    
  ask patches with [ pond-area > 0 ]
    [ 
      set pcolor blue
      set depth 0.2
      set cover 0
    ]
end

to setup-patches-random
  ask patches                          ; set-up upland forest patches
  [
    set cover random 100
    set temp-cover (random forest-cover-multiplier)
  ]
  if forest-cover-multiplier > 0
  [
    ask patches with [ cover < (100 - temp-cover) ]
        [ set cover cover + temp-cover ]
  ]
  if forest-cover-multiplier < 0
  [
    ask patches with [ cover > (100 - (100 + temp-cover)) ]
      [ set cover cover + temp-cover ] 
  ]
  ask patches
  [
    set pcolor (59 - cover / 13)       ; lighter colors mean less cover
  ]

  ask n-of number-ponds patches        ; set up pond patch
  [
    set cover 0
    set pcolor blue
    set depth 0.2                      ; unit meters
    set cat-area-random ( random-normal catchment-area (random catchment-area) )  ; user inputs the average catchment size of ponds (in patches)
    set pond-area-random  1                                                 ; all ponds assumed to have area of 1 patch
    set pond-depths [ depth ] of patches with [ pcolor = blue ] 
  ]
end

to-report read-rain-file [ file-name ]
  file-open file-name
  let temp file-read-characters 6           ; print temp
  set rain-data-points file-read                   ;; denote the numbers of rainfall data records
  print rain-data-points
  let array []
  let r_count 0
  while [ r_count < rain-data-points ]
  [
    set array lput file-read array
    set r_count r_count + 1
  ]
  file-close
  report array
end

to choose-rainfall                                                  ; the user-selected rainfall dataset is used to run the model
  if precipitation = "observed"          [ set rainfall array:from-list read-rain-file "input/P_observed" ]
  if precipitation = "30%_observed"      [ set rainfall array:from-list read-rain-file "input/P_observed_dry0.3" ]
  if precipitation = "60%_observed"      [ set rainfall array:from-list read-rain-file "input/P_observed_dry0.6" ]
  if precipitation = "high_seasonal"     [ set rainfall array:from-list read-rain-file "input/P_wet_winter" ]
  if precipitation = "low_seasonal"      [ set rainfall array:from-list read-rain-file "input/P_dry_summer" ]
  if precipitation = "high_inter-annual" [ set rainfall array:from-list read-rain-file "input/P_wet_all_the_year" ]
  
  set past-precip-1 array:item rainfall (rain-data-points - 1)              ; the initial values for previous 5 months' rainfall are taken from end of rainfall dataset
  set past-precip-2 array:item rainfall (rain-data-points - 2)
  set past-precip-3 array:item rainfall (rain-data-points - 3)
  set past-precip-4 array:item rainfall (rain-data-points - 4)
  set past-precip-5 array:item rainfall (rain-data-points - 5)
end

to setup-newts
  create-newts initial-newts
  ask newts
  [
    set shape "newt"
    set size .7
    set color red
    set life-stage "adult"
    set rain-sens random-normal rainfall-threshold threshold-SD  ; set user-defined newt response to rainfall
    set total-breed-depth 0                                             ; set counters used to track breeding
    set breeding-counter 6
    set mort-multiplier upland-mortality                                ; set the mortality rate to the user-defined mortality adjustment - mortality rates do not fluctuate during a model run
                                                                        ; EXCEPT when the user activates precipitation-based mortality. (see mortality procedure, below)
  ]
end

to setup-newts-upland            ; This function places the newts randomly in the upland, with a greater chance of appearing in patches with high cover
                                 ; and near to ponds.
  ask newts
  [ 
    let random-dist 0
    let new-patch nobody
    while [ new-patch = nobody ]
    [
      let random-cover random 10000
      ifelse random 2 > 0
      [ set random-dist 3 + sqrt random 10 ]
      [ set random-dist 3 - sqrt random 10 ]
      set new-patch one-of patches with [ (cover ^ 2 > random-cover) and ( distance min-one-of ( patches with [ pcolor = blue ] ) [ distance myself ] < random-dist ) ]
      if NOT ( new-patch = nobody ) [ move-to new-patch ]
    ]
    set home-pond-x [ pxcor ] of min-one-of ( patches with [ pcolor = blue ] ) [ distance myself ]
    set home-pond-y [ pycor ] of min-one-of ( patches with [ pcolor = blue ] ) [ distance myself ]
    forward random-float .5
    set trigger "upland"
  ]
end

to setup-efts
  set eft-proportion .3                                            ; we assume that 30% of newts begin as efts.
  ask newts with [ int (remainder who (1 / eft-proportion)) = 0 ]
  [
    set life-stage "eft"
    set color violet
    set eft-counter random 12
    set home-pond-x "none"
  ]
end


;**************************************************************************************************************************************
;**************************************************************************************************************************************

to GO
;  if ticks >= 287 [ stop ]
  rain-event
  change-depth
  track-precip-av
  move-newts-upland
  move-newts-pond
  track-distance
  efts-age
  reproduce
  if count newts <= 0 [stop]
  mortality
  do-plots
  tick
end


to change-depth                          ; Sets the depths of all ponds in the current time-step
  set ppt (array:item rainfall ( ticks mod rain-data-points ) )
  let evap 0
  
  ask patches with [ pcolor = blue ]
  [
    ifelse EvapoT_Proportion?             ; the user chooses whether to use a depth-dependent evapo-transpiration function with a user-inputted transpiration rate
                                          ; or a constant rate derived from the literature.
      [ set evap depth * EvapoT-proportion ]
      [ set evap 15 * 11.36 * 1.3 / 1000  ]  ; NEED TO PROVIDE A REFERENCE FOR THESE VALUES.
    let leak depth * (drainage-proportion / 100)

    ifelse Real_Landscape?                ; mechanism is slightly different in Real and Simulated landscapes. In the real landscape, pond and catchment surface areas are read-in from GIS data.
                                          ; In the simulated landscape, catchment areas are user-generated, and all ponds are assumed to be 100x100 m in area.
    [    
      set depth  depth + ppt / 1000 * cat-area / pond-area * 0.7 - evap - leak
      if depth < 0 [  set depth 0  ]
    ]
    [
      set depth  depth + ppt / 1000 * cat-area-random / pond-area-random - evap - leak  ; depth is a function of a) depth in the previous month, b) runoff from the
                                                                                        ; catchment area (ppt/1000*cat-area-random/pond-area-random), c) evapo-transpiration, and
                                                                                        ; d) leakage/percolation.
      if depth < 0 [  set depth 0  ]
    ]
    set pond-depths [ depth ] of patches with [ pcolor = blue ]  
  ]
end


to rain-event
 if ticks > 0
 [
   ask newts with [ (trigger = "upland") and ( ppt >= rain-sens) ]    ; newts in upland repond to monthly rain falls greater than their rain sensitivity by returning to home ponds.
   [
     set trigger "migrate-pond"
   ]
 ]
end

to move-newts-upland                             ; Newts with trigger "migrate-up" will move into the upland, favoring patches with high cover.
  clear-drawing                ; clears the traces that newts draw when moving
  ask newts with [trigger = "migrate-up"]
  [
    let my-color color
    set color black
    pen-down
    let settled FALSE
    while [ NOT settled ]
    [
      right random 360
      forward 1
      if ((cover ^ 2) > random 10000) AND count newts-here < upland-capacity            ; If cover of patch is greater than a random number, they end migration. If not, they continue.
      [
        set trigger "upland"
        set settled TRUE
      ]
    ]
    pen-up
    set color my-color
  ]
end

to move-newts-pond
  ask newts with [ trigger = "migrate-pond" and life-stage = "adult"]    ; Adults with trigger "migrate-pond" return to home ponds to breed, without suffering mortality.
  [
    move-to patch home-pond-x home-pond-y
    set trigger "breed"
    set breeding-counter 6
  ]
end


to track-distance
  ask newts with [home-pond-x != "none"] [ set distance-to-pond ( distancexy home-pond-x home-pond-y ) ]    ; track distance from home pond
end


to efts-age              ; When they move to the upland, efts make the closest pond their home pond, and wait 12 months to mature to adults
  ask newts with [ life-stage = "eft" and home-pond-x = "none"]
  [
    set home-pond-x [ pxcor ] of min-one-of ( patches with [ pcolor = blue ] ) [ distance myself ]
    set home-pond-y [ pycor ] of min-one-of ( patches with [ pcolor = blue ] ) [ distance myself ]
  ]
  ask newts with [ life-stage = "eft" and eft-counter > 0]
  [
    set eft-counter eft-counter - 1
  ]
  ask newts with [ life-stage = "eft" and eft-counter = 0]
  [
    set life-stage "adult"
    set color red
  ]
end

to reproduce                                  ; Adults breeding in ponds take six months to produce efts. If pond depth drops to 0 during this
                                              ; period, they fail to breed and return to the upland. The number of efts produced by an individual
                                              ; is a function of average water in pond during 6 months, the number of newts in the pond, and user-inputted
                                              ; maximum reproductive rate (defined as max. number of efts with no competition and 1 m pond depth).
                                              
  ask newts with [ (life-stage = "adult") and (trigger = "breed")]
  [
    ifelse breeding-counter > 0
    [
      ifelse depth > 0
      [
        set breeding-counter (breeding-counter - 1)
        set total-breed-depth total-breed-depth + depth
      ]
      [
        set trigger "migrate-up"
        set breeding-counter 6
        set total-breed-depth 0
      ]
    ]
    [
      hatch ( ((total-breed-depth + .0001)/ 6) * (random (max-recruitment )) * (1 - ((count newts-here) / pond-capacity )))
      [
        set color violet                                                      ; Creates efts and assigns them traits.
        set life-stage "eft"
        set trigger "migrate-up"
        set rain-sens random-normal rainfall-threshold threshold-SD
        set breeding-counter 6
        set total-breed-depth 0
        set eft-counter 12
        set home-pond-x "none"
      ]
      set trigger "migrate-up"
      set breeding-counter 6
      set total-breed-depth 0
    ]   
  ]
end

to track-precip-av                      ; Tracks the six month average precipitation (including current month) for the purpose of determining the impact of rainfall on upland mortality.

  set precip-av ((ppt + past-precip-1 + past-precip-2 + past-precip-3 + past-precip-4 + past-precip-5) / 6)
  set past-precip-5 past-precip-4
  set past-precip-4 past-precip-3
  set past-precip-3 past-precip-2
  set past-precip-2 past-precip-1
  set past-precip-1 ppt  
end


to mortality                                    ; Mortality occurs only in the upland. It is a function of 3 or 4 factors, depending on whether the user activates precipitation-based
                                                ; mortality. 1) the cover of the patch a newt occupies, 2) the number of newts in the patch (carrying capacity is user-defined),
                                                ; 3) user-defined mortality-multiplier (upland-mortality). If the user has activated precipitation-based mortality, the fourth factor is:
                                                ; 4) quotient of the user-defined base precipitation rate (precip-upland-mortality) and the six-month average rainfall.
  if precip_mortality?
    [ set mort-multiplier upland-mortality * sqrt sqrt (precip-upland-mortality / precip-av) ]

  ask newts with [pcolor != blue]
  [
    if random 10 * upland-mortality * mort-multiplier > ((sqrt cover) * (1 - ((count newts-here) / upland-capacity ))) [ die ]
  ]
end

to do-plots
if count newts > 0
[
  set-current-plot "Distribution"  
    histogram [ distance-to-pond ] of newts      ;; draws a histogram of number of newts at given distance
  set-current-plot "distance to pond"
    set-current-plot-pen "max-dist"
      plot [distance-to-pond] of max-one-of newts [distance-to-pond]
    set-current-plot-pen "av-dist"
      plot mean [distance-to-pond] of newts  
  set-current-plot "pond depth"
    set-current-plot-pen "depth"
      plot mean [depth] of patches with [ pcolor = blue ]
  set-current-plot "rainfall"
    set-current-plot-pen "rainfall"
      plot ppt
    set-current-plot-pen "6 month average"
      plot precip-av
  set-current-plot "Number of newts"
    set-current-plot-pen "adults"
      plot count newts with [ life-stage = "adult" ]
    set-current-plot-pen "efts"
      plot count newts with [ life-stage = "eft" ]
  set-current-plot "Mortality Rate"
    set-current-plot-pen "precip mortality factor"
      plot mort-multiplier
]
end
@#$#@#$#@
GRAPHICS-WINDOW
604
10
1374
881
-1
-1
40.0
1
10
1
1
1
0
0
0
1
0
18
0
20
1
1
1
months
30.0

BUTTON
295
35
361
68
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
364
35
427
68
Step
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
434
35
489
68
Go
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
222
755
598
875
Distribution
distance pond
number
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"Newts" 1.0 1 -10899396 true "" ""

PLOT
222
630
599
751
Distance to pond
time
distance
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"av-dist" 1.0 0 -10899396 true "" ""
"Max-dist" 1.0 0 -11221820 true "" ""

SLIDER
16
293
204
326
initial-newts
initial-newts
1
1000
266
5
1
newts
HORIZONTAL

PLOT
222
498
598
624
Pond Depth
NIL
meters
0.0
10.0
0.0
0.5
true
false
"" ""
PENS
"depth" 1.0 0 -13791810 true "" ""
"pond1" 1.0 0 -7500403 true "" ""
"pond2" 1.0 0 -2674135 true "" ""
"pond3" 1.0 0 -955883 true "" ""
"pond4" 1.0 0 -6459832 true "" ""
"pond5" 1.0 0 -1184463 true "" ""
"pond6" 1.0 0 -10899396 true "" ""

SLIDER
11
133
188
166
number-ponds
number-ponds
0
15
9
1
1
NIL
HORIZONTAL

PLOT
220
100
596
235
Number of newts
time
Individuals
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"adults" 1.0 0 -2674135 true "" ""
"efts" 1.0 0 -8630108 true "" ""

SLIDER
15
590
201
623
upland-mortality
upland-mortality
.5
1.5
0.94
.01
1
NIL
HORIZONTAL

SLIDER
16
409
219
442
max-recruitment
max-recruitment
1
40
31
1
1
per ind.
HORIZONTAL

SLIDER
15
482
205
515
pond-capacity
pond-capacity
0
500
110
5
1
newts
HORIZONTAL

SLIDER
11
170
189
203
catchment-area
catchment-area
.2
10
5
.2
1
patches
HORIZONTAL

SLIDER
16
332
206
365
rainfall-threshold
rainfall-threshold
0
300
70
5
1
mm
HORIZONTAL

SLIDER
16
370
206
403
threshold-SD
threshold-SD
0
200
60
5
1
mm
HORIZONTAL

CHOOSER
10
31
201
76
Precipitation
Precipitation
"observed" "30%_observed" "60%_observed" "high_seasonal" "low_seasonal" "high_inter-annual"
0

PLOT
222
238
598
372
Rainfall
time
mm
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"rainfall" 1.0 0 -13345367 true "" ""
"6 month average" 1.0 0 -2064490 true "" ""

SLIDER
11
209
189
242
forest-cover-multiplier
forest-cover-multiplier
-100
100
0
5
1
%
HORIZONTAL

SLIDER
15
665
202
698
precip-upland-mortality
precip-upland-mortality
1
120
111
5
1
NIL
HORIZONTAL

SWITCH
11
100
162
133
Real_Landscape?
Real_Landscape?
0
1
-1000

SLIDER
15
736
210
769
drainage-proportion
drainage-proportion
0
150
136
2
1
%
HORIZONTAL

TEXTBOX
16
713
166
732
Pond Parameters\n
15
0.0
1

SWITCH
13
778
203
811
EvapoT_Proportion?
EvapoT_Proportion?
1
1
-1000

SLIDER
13
812
203
845
evapoT-proportion
evapoT-proportion
0
150
50
2
1
%
HORIZONTAL

TEXTBOX
11
78
256
97
Simulated Landscape Params
14
0.0
1

TEXTBOX
26
273
204
292
Newt Variables
14
0.0
1

SLIDER
15
520
205
553
upland-capacity
upland-capacity
10
2000
10
20
1
newts/ha
HORIZONTAL

TEXTBOX
19
458
169
476
Carrying Capacities
14
0.0
1

SWITCH
15
632
202
665
Precip_Mortality?
Precip_Mortality?
0
1
-1000

TEXTBOX
15
568
165
586
Mortality Parameters
14
0.0
1

TEXTBOX
14
10
164
28
Climate Scenario
14
0.0
1

PLOT
222
373
598
493
Mortality Rate
NIL
NIL
0.0
10.0
0.5
1.5
true
false
"" ""
PENS
"precip mortality factor" 1.0 0 -16777216 true "" ""

MONITOR
538
40
596
85
Newts
count newts
0
1
11

@#$#@#$#@
## WHAT IS THE MODEL ABOUT?

This model explores how potential behaviors and ecological insteractions of individual striped newts (Notophthalmus perstriatus) affect population distributions and viability.  The model operates in the context of a landscape defined by ponds and habitat quality. Because little is know about striped newt ecology, this model is intended to help investigate the importance of a set of ecological responses and landscape states to newt population survival.

The variables used in the model fall into two categories: landscape and newt properties. Landscape properties include precipitation regime, upland habitat quality, density of breeding ponds, pond and upland carrying capacities, and pond hydrologies. Newt properties include mortality, maximum potential recruitment rate, and the rainfall threshold that triggers breeding.

The model aims to contribute to the identification of processes and variables that are priorities for future field research and that importantly influence the viability of striped newt populations.

The following sections describe the use of the model, and the agents, patches, and processes of which the model is composed. For more detailed information on the data and mechanisms used in the model, the user should refer to the accompanying report: �Spatially-explicit agent-based model of striped-newt (Notophthalmus perstriatus) metapopulation dynamics under precipitation and forest cover scenarios.�

## HOW TO USE THE MODEL

1. Before running the model, select starting conditions: (see �Setup Operation� for further description)  
a)	climate scenario � the rainfall scenario to be used  
b)	real landscape (from Fort Stewart, GA) OR a simulated, hypothetical landscape.

2. If desired, alter initial variables:  
a)	"initial-newts"
           - total number of newts initialized to start the model

b)	"rainfall-threshold" � the minimum amount of rainfall required for newts to return to their home ponds to reproduce.  
c)	"threshold-SD"
          - the standard deviation of the rainfall threshold, distributed across newts

d)	"max-recruitment"
        - The maximum number of efts that can be produced by a single adult in one breeding event

3. If desired, turn on optional mechanisms:  
a)	precipitation-based mortality (as opposed to constant mortality rate) � when �on� the mortality rate in uplands will vary with the six-month mean rainfall.  
b)	proportional evapo-transpiration (as opposed to a constant rate from the literature) � when �on,� a fixed proportion of water evaporates from ponds each month.

4. Hit "Setup" to initialize the newts and landscape.

5. Hit "step" to advance the model by a one-month time step OR hit "go" for a continuous analysis.

6. Six variables can be modified at any time during the model run:  
a)
      "pond-capacity"           - number of newts per 1Ha pond area

b)	"upland-capacity"
        - number of newts per 1Ha upland area

c)	"upland-mortality"
      - alters the mortality rate across the entire model run

d)	"precip-upland-mortality" -  alters the mortality rate according to the six-month mean of rainfall (only when Precip_Mortality? is �on�)  
e)	"Drainage-proportion"
               - the percent of water that percolates from ponds each month

f)	"EvapoT-proportion"
               - the percent of water that evaporates each month (only when EvapoT_proportion? is �on�)


## AGENTS AND PATCHES

Newts are individual agents that live and reproduce on the landscape.  Time steps are months.  Each month a newt either migrates-up into the uplands, migrates to its home pond, reproduces, or remains in the uplands.  Newts remember the location of their home ponds and return to these ponds to breed when the monthly rainfall exceeds a user-defined threshold (rainfall-threshold).

Patches are 100x100m (1 hectare) in size. The user selects one of two methods for generating the model landscape:

1.	The first method generates a landscape based on a quandrant in the northwest of the Fort Stewart army installation, Georgia. In this method, GIS maps are read-in to NetLogo to provide the following information for each patch:  
a) whether patches are depressions that form ephemeral ponds  
b) percentage forest cover of upland patches (a measure of habitat quality)  
c) surface area occupied by ponds when filled (in m2)  
d) area of pond catchments (in m2)

2.	The second method generates a simulated method based on user-determined variables:  
a) the number of ponds in the landscape frame  
b) the average catchment area draining to each pond  
c) the distribution of habitat quality (forest cover) based on a user-determined multiplier (see the model report for more details on the mechanism through which habitat quality is generated)

During model execution, the following state variables are managed:  
- water depth in ponds  
- amount of area draining into each pond  
- whether patch is a depression

## SETUP OPERATION

The following variables alter the initial conditions in the model:

a) initial-newts - the starting number of newts

b) precipitation - the �Rainfall and Hydrology� section describes each of the six precipitation scenarios

c) Real_Landscape? � �On� means that GIS maps of an area in Fort Stewart, GA are used to generate the landscape.  �Off� means that a simulated landscape is generated. The user selects the number of ponds, mean size of catchment area for the ponds, a forest cover multiplier.

## GO OPERATION

In each month-long time step, the following processes occur:

1.	Rain occurs based on the user-selected rain schedule (the "observed" is based on a 24 year Fort Stewart rain history)

2.	The depth of water in ponds is adjusted due to rainfall, evaporation, and percolation.

3.	Newts leaving ponds migrate to the upland areas.

4.	Newts whose rainfall threshold is exceeded return to their home ponds to breed.

5.	Efts age (after 12 months they become adults).

6.	Adults breeding in ponds take six months to produce efts.

7.	Efts in the uplands die.


## NEWT BEHAVIOR AND REPRODUCTION

1. Eft Lifestage  
Because few data were available on striped newt reproduction and development, egg, larvae and paedomorph life stages are not considered in the model. Instead, newts are "born" as juvenile efts. Efts immediately move into upland areas surrounding their natal ponds, where they remain as efts for 12 months.

2. Newt movement  
Each newt is assigned an individual rainfall threshold value that defines the minimum monthly rainfall required to trigger reproductive attempts. Rainfall thresholds are determined based on a user-selected average and standard deviation. Following reproductive attempts, newts move out of ponds in a path determined by the habitat quality of each patch they encounter. Newts are more likely to enter a patch with higher quality, and more likely to remain in this patch once they have entered.

3. Reproduction  
Newts respond to sufficiently large rainfall by returning to home ponds to reproduce, as described above. Newts that are reproducing remain in the pond for a 6-month period, during which each newt stores the average pond depth value. If pond depth drops to 0 during the breeding period, newts abandon reproduction and move into the upland without producing efts. When reproduction is successful, the number of efts �born� per individual is a function of three factors:  
a)	average pond depth during the 6-month breeding period  
b)	the number of newts present in the pond at the end of breeding  
c)	a user-inputted maximum reproductive rate (defined as max. number of efts produced by an adult with no competition and 1 m pond depth)

4.	Mortality  
Mortality occurs in each time-step for all newts that are present in the upland. Mortality is always the product of three factors:  
a) the cover of the patch a newt occupies  
b) the number of newts in the patch (carrying capacity is user-defined)  
c) user-defined mortality-multiplier (upland-mortality)  
A fourth factor impacts mortality only when the user activates precipitation-based mortality. The user should activate precipitation-based mortality if (s)he believes precipitation is important in determining upland habitat quality.  
d) the quotient of the base precipitation rate and the six-month average rainfall. 

## HYDROLOGY AND RAINFALL

Hydrology in the model defines the relationship between the user-selected precipitation scenario and pond depths. The model assumes that ponds are �buckets� filled by water running off catchment areas. The model excludes evapo-transpiration and percolation as water flows off upland patches. Pond depth is defined as depth in the previous month, plus run-off, minus evapo-transpiration and percolation/drainage in the pond.

The model assumes that the percolation/drainage rate is a linear function of pond depth. The user must select the drainage proportion, defined as the proportion of depth in each pond that drains in a given time step.

The user can select between two methods for calculating evapo-transpiration rates in ponds:  
a)	a fixed rate derived from the literature  
b)	the same mechanism as drainage - the user selects the proportion of water evaporating each month.

The following rainfall scenarios are available in the model, and need to be chosen before hitting the setup button:  
a)	observed (based on a 24-year precipitation record at Fort Stewart (1985-2008))  
b)	30% observed (observed rainfall in each month is multiplied by 30%)  
c)	60% observed  
d)	high seasonal variability (the total rainfall falling over the study period is the same as in the observed scenario, but month-to-month variation is increased.)  
e)	low seasonal variability (same total rainfall, less month-to-month variation)  
f)	high inter-annual variability (same total rainfall, greater year-to-year variation)

## GIS INPUT DATA

The following GIS maps (arc ascii grids) are read in at system startup:  
a)	canopy_100_cubic.asc  - Provides percent tree canopy values. (Used as a measure of habitat quality for determining newt movement and mortality)  
b)	ponds_100_3.asc - Gives the total area of the pond at this point.  
c)	catchment_100_3.asc - Gives the total catchment area of the pond at this point (These maps are used with rainfall information to fill and drain ponds).

## EXTENDING THE MODEL

Relatively few data were available to inform the development of this model. The mechanisms determining newt behavior and survival are thus broad, and depend on a relatively small set of ecological variables. Since there are few quantitative estimates of ecological variables, the user inputs the majority based on her/his understanding of newt ecology.

The model might be extended by improving the accuracy of data and mechanisms, including but not limited to the following changes:

1)	Calibration and sensitivity analysis based on additional data.  
Where available, model behavior could be calibrated against empirical data on striped newt populations. For example, existing studies measure rates of newt migration to and from ponds and monthly rainfall over short time periods (Dodd 1993, Johnson 2002). Rainfall and landscape data from these studies could be input into the model, and newt migration rates simulated and compared against study data.

2)	More detailed and realistic climate scenarios.  
The precipitation scenarios used in this report represent initial, ideal �types� of change in rainfall quantity and variability. More sophisticated scenarios, including those incorporating the projected impacts of climate change, could easily be substituted.

3)	Alternative measures of habitat quality.  
Forest canopy cover was used as a representation of habitat quality because the data were readily available for Fort Stewart. Ground cover characteristics, and other variables, may be more important in determining newt movement and mortality. Other measure(s) of habitat quality could be substituted using GIS to map the distribution of quality across the landscape. The mortality and movement functions would need to be altered if multiple measures are used, or if the measure is based on a range other than 0-100.

4)	More detailed hydrological model.  
This model assumed a relatively simple hydrological model. More accurate data on run-off, percolation and evapo-transpiration could be used in conjunction with the model GIS layers to enhance the sophistication of the hydrological model. The hydrology of the simulated model might be similarly improved.

5)	Data on pond-level processes (reproduction, larval development, competition, mortality).  
The model assumes a simplified reproductive processes in which newts are recruited directly as efts. If data were available on the pond-level processes that shape reproduction, the model could generate more accurate estimates of eft out-migration following breeding events. Initial sensitivity analysis suggests, however, that pond-level processes may not be the most important factors determining newt population dynamics (see the accompanying report).

Many further calibrations and improvements could be made where data are available.

## CREDITS AND REFERENCES

Developed by Sheng Ye, Ewan Robinson, and Jennifer Burton (DVM), all graduate students in the department of Geography at the University of Illinois at Urbana-Champaign. Completed as part of the Geography 496 "Spatial Ecological Modeling" course taught by Dr. Bruce Hannon and Dr. Jim Westervelt in Spring 2009. Direct communications to Jim Westervelt at
           		

westerve@illinois.edu.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

ant
true
0
Polygon -7500403 true true 136 61 129 46 144 30 119 45 124 60 114 82 97 37 132 10 93 36 111 84 127 105 172 105 189 84 208 35 171 11 202 35 204 37 186 82 177 60 180 44 159 32 170 44 165 60
Polygon -7500403 true true 150 95 135 103 139 117 125 149 137 180 135 196 150 204 166 195 161 180 174 150 158 116 164 102
Polygon -7500403 true true 149 186 128 197 114 232 134 270 149 282 166 270 185 232 171 195 149 186
Polygon -7500403 true true 225 66 230 107 159 122 161 127 234 111 236 106
Polygon -7500403 true true 78 58 99 116 139 123 137 128 95 119
Polygon -7500403 true true 48 103 90 147 129 147 130 151 86 151
Polygon -7500403 true true 65 224 92 171 134 160 135 164 95 175
Polygon -7500403 true true 235 222 210 170 163 162 161 166 208 174
Polygon -7500403 true true 249 107 211 147 168 147 168 150 213 150

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

newt
true
0
Polygon -7500403 true true 186 188 197 193 193 199 177 198 164 195 167 183
Polygon -7500403 true true 165 87 181 85 188 85 195 80 200 74 207 79 197 91 186 96 167 101
Polygon -7500403 true true 144 86 128 87 115 84 105 80 98 88 107 91 115 95 127 99 143 101
Polygon -7500403 true true 142 99 138 68 129 46 127 21 142 7 159 13 165 37 165 65 166 95
Polygon -7500403 true true 147 183 135 186 122 195 127 204 141 194 149 190
Polygon -7500403 true true 143 99 168 94 175 147 163 201 152 235 142 283 135 300 136 267 144 221 150 163

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
